<!DOCTYPE html>
<html lang="hr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - TapTap Card</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --danger: #ef4444;
            --warning: #f59e0b;
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-400: #9ca3af;
            --gray-500: #6b7280;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: var(--gray-900);
        }
        
        .container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar */
        .sidebar {
            width: 240px;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 24px 0;
            display: flex;
            flex-direction: column;
            box-shadow: 2px 0 20px rgba(0, 0, 0, 0.1);
        }
        
        .logo {
            padding: 0 24px 24px;
            font-size: 24px;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .nav-menu {
            flex: 1;
            padding: 0 12px;
        }
        
        .nav-item {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px 16px;
            margin-bottom: 4px;
            border-radius: 10px;
            color: var(--gray-600);
            text-decoration: none;
            font-weight: 500;
            font-size: 15px;
            transition: all 0.2s;
            cursor: pointer;
        }
        
        .nav-item:hover {
            background: var(--gray-100);
            color: var(--gray-900);
        }
        
        .nav-item.active {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
        }
        
        .nav-item svg {
            width: 20px;
            height: 20px;
        }
        
        .user-info {
            padding: 16px 24px;
            border-top: 1px solid var(--gray-200);
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 16px;
        }
        
        .user-details {
            flex: 1;
            min-width: 0;
        }
        
        .user-name {
            font-weight: 600;
            font-size: 14px;
            color: var(--gray-900);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .user-role {
            font-size: 12px;
            color: var(--gray-500);
        }
        
        .logout-btn {
            background: none;
            border: none;
            color: var(--gray-400);
            cursor: pointer;
            padding: 4px;
            display: flex;
            align-items: center;
        }
        
        .logout-btn:hover {
            color: var(--danger);
        }
        
        /* Main Content */
        .main-content {
            flex: 1;
            padding: 32px;
            overflow-y: auto;
        }
        
        .page-header {
            margin-bottom: 32px;
        }
        
        .page-title {
            font-size: 32px;
            font-weight: 700;
            color: white;
            margin-bottom: 8px;
        }
        
        .page-subtitle {
            color: rgba(255, 255, 255, 0.8);
            font-size: 16px;
        }
        
        /* Cards Grid */
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 24px;
            margin-bottom: 32px;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 24px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        .card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .card-title {
            font-size: 14px;
            font-weight: 600;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .card-action {
            color: var(--primary);
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 4px;
        }
        
        .card-action:hover {
            color: var(--primary-dark);
        }
        
        /* Stats Card */
        .stat-value {
            font-size: 36px;
            font-weight: 700;
            color: var(--gray-900);
            margin-bottom: 8px;
        }
        
        .stat-label {
            font-size: 14px;
            color: var(--gray-500);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            margin-bottom: 16px;
        }
        
        .stat-icon.primary {
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
        }
        
        .stat-icon.success {
            background: rgba(16, 185, 129, 0.1);
        }
        
        .stat-icon.warning {
            background: rgba(245, 158, 11, 0.1);
        }
        
        .stat-change {
            font-size: 13px;
            font-weight: 600;
            padding: 4px 8px;
            border-radius: 6px;
        }
        
        .stat-change.up {
            color: var(--success);
            background: rgba(16, 185, 129, 0.1);
        }
        
        .stat-change.down {
            color: var(--danger);
            background: rgba(239, 68, 68, 0.1);
        }
        
        /* Profile Preview Card */
        .profile-preview {
            display: flex;
            gap: 24px;
            align-items: start;
        }
        
        .profile-avatar-large {
            width: 80px;
            height: 80px;
            border-radius: 16px;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 32px;
            font-weight: 700;
            flex-shrink: 0;
        }
        
        .profile-info {
            flex: 1;
        }
        
        .profile-name {
            font-size: 24px;
            font-weight: 700;
            color: var(--gray-900);
            margin-bottom: 4px;
        }
        
        .profile-desc {
            color: var(--gray-600);
            font-size: 15px;
            margin-bottom: 12px;
        }
        
        .profile-location {
            display: flex;
            align-items: center;
            gap: 6px;
            color: var(--gray-500);
            font-size: 14px;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 20px;
            border-radius: 10px;
            font-weight: 600;
            font-size: 14px;
            border: none;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: white;
            color: var(--gray-700);
            border: 2px solid var(--gray-200);
        }
        
        .btn-secondary:hover {
            border-color: var(--gray-300);
            background: var(--gray-50);
        }
        
        /* QR Code Card */
        .qr-container {
            text-align: center;
        }
        
        .qr-code {
            width: 200px;
            height: 200px;
            background: white;
            border: 2px solid var(--gray-200);
            border-radius: 12px;
            margin: 0 auto 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 80px;
        }
        
        .share-buttons {
            display: flex;
            gap: 8px;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .share-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 13px;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        /* Contacts Table */
        .contacts-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .contacts-table th {
            text-align: left;
            padding: 12px;
            font-size: 12px;
            font-weight: 600;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid var(--gray-200);
        }
        
        .contacts-table td {
            padding: 16px 12px;
            border-bottom: 1px solid var(--gray-100);
        }
        
        .contact-avatar {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 14px;
        }
        
        .contact-name {
            font-weight: 600;
            color: var(--gray-900);
            margin-bottom: 2px;
        }
        
        .contact-email {
            font-size: 13px;
            color: var(--gray-500);
        }
        
        .tag {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 500;
            background: var(--gray-100);
            color: var(--gray-600);
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray-400);
        }
        
        .empty-state-icon {
            font-size: 64px;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        
        /* Loading */
        .loading {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 40px;
            color: var(--gray-400);
        }
        
        .spinner {
            border: 3px solid var(--gray-200);
            border-top-color: var(--primary);
            border-radius: 50%;
            width: 32px;
            height: 32px;
            animation: spin 0.8s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Mobile */
        @media (max-width: 768px) {
            .sidebar {
                position: fixed;
                left: -240px;
                top: 0;
                height: 100vh;
                z-index: 1000;
                transition: left 0.3s;
            }
            
            .sidebar.open {
                left: 0;
            }
            
            .main-content {
                padding: 20px;
            }
            
            .grid {
                grid-template-columns: 1fr;
            }
            
            .profile-preview {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="logo">wave</div>
            
            <nav class="nav-menu">
                <a href="#" class="nav-item active" data-page="profile">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path d="M10 9a3 3 0 100-6 3 3 0 000 6zm-7 9a7 7 0 1114 0H3z"/>
                    </svg>
                    Profile
                </a>
                
                <a href="#" class="nav-item" data-page="contacts">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path d="M9 6a3 3 0 11-6 0 3 3 0 016 0zM17 6a3 3 0 11-6 0 3 3 0 016 0zM12.93 17c.046-.327.07-.66.07-1a6.97 6.97 0 00-1.5-4.33A5 5 0 0119 16v1h-6.07zM6 11a5 5 0 015 5v1H1v-1a5 5 0 015-5z"/>
                    </svg>
                    Contacts
                </a>
                
                <a href="#" class="nav-item" data-page="analytics">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path d="M2 11a1 1 0 011-1h2a1 1 0 011 1v5a1 1 0 01-1 1H3a1 1 0 01-1-1v-5zM8 7a1 1 0 011-1h2a1 1 0 011 1v9a1 1 0 01-1 1H9a1 1 0 01-1-1V7zM14 4a1 1 0 011-1h2a1 1 0 011 1v12a1 1 0 01-1 1h-2a1 1 0 01-1-1V4z"/>
                    </svg>
                    Analytics
                </a>
                
                <a href="#" class="nav-item" data-page="settings">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M11.49 3.17c-.38-1.56-2.6-1.56-2.98 0a1.532 1.532 0 01-2.286.948c-1.372-.836-2.942.734-2.106 2.106.54.886.061 2.042-.947 2.287-1.561.379-1.561 2.6 0 2.978a1.532 1.532 0 01.947 2.287c-.836 1.372.734 2.942 2.106 2.106a1.532 1.532 0 012.287.947c.379 1.561 2.6 1.561 2.978 0a1.533 1.533 0 012.287-.947c1.372.836 2.942-.734 2.106-2.106a1.533 1.533 0 01.947-2.287c1.561-.379 1.561-2.6 0-2.978a1.532 1.532 0 01-.947-2.287c.836-1.372-.734-2.942-2.106-2.106a1.532 1.532 0 01-2.287-.947zM10 13a3 3 0 100-6 3 3 0 000 6z"/>
                    </svg>
                    Settings
                </a>
            </nav>
            
            <div class="user-info">
                <div class="user-avatar" id="sidebarAvatar">U</div>
                <div class="user-details">
                    <div class="user-name" id="sidebarName">Loading...</div>
                    <div class="user-role">User</div>
                </div>
                <button class="logout-btn" onclick="logout()">
                    <svg width="20" height="20" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M3 3a1 1 0 00-1 1v12a1 1 0 102 0V4a1 1 0 00-1-1zm10.293 9.293a1 1 0 001.414 1.414l3-3a1 1 0 000-1.414l-3-3a1 1 0 10-1.414 1.414L14.586 9H7a1 1 0 100 2h7.586l-1.293 1.293z"/>
                    </svg>
                </button>
            </div>
        </aside>
        
        <!-- Main Content -->
        <main class="main-content">
            <!-- Profile Page -->
            <div id="profilePage">
                <div class="page-header">
                    <h1 class="page-title">Welcome back! 👋</h1>
                    <p class="page-subtitle">Here's what's happening with your profile</p>
                </div>
                
                <!-- Analytics Mini Cards -->
                <div class="grid">
                    <div class="card">
                        <div class="stat-icon primary">👁️</div>
                        <div class="stat-value" id="statPageViews">-</div>
                        <div class="stat-label">
                            Page Views
                            <span class="stat-change up" id="viewsChange" style="display: none;">↑ 12%</span>
                        </div>
                    </div>
                    
                    <div class="card">
                        <div class="stat-icon success">✨</div>
                        <div class="stat-value" id="statClicks">-</div>
                        <div class="stat-label">
                            Page Clicks
                            <span class="stat-change up" id="clicksChange" style="display: none;">↑ 8%</span>
                        </div>
                    </div>
                    
                    <div class="card">
                        <div class="stat-icon warning">📊</div>
                        <div class="stat-value" id="statCTR">-</div>
                        <div class="stat-label">
                            CTR
                            <span class="stat-change" id="ctrChange" style="display: none;">-</span>
                        </div>
                    </div>
                </div>
                
                <!-- Profile & QR -->
                <div class="grid">
                    <div class="card" style="grid-column: 1 / -1;">
                        <div class="card-header">
                            <h3 class="card-title">Your Profile</h3>
                            <a href="user-panel.php" class="card-action">
                                Edit Profile
                                <svg width="16" height="16" fill="currentColor" viewBox="0 0 20 20">
                                    <path d="M13.586 3.586a2 2 0 112.828 2.828l-.793.793-2.828-2.828.793-.793zM11.379 5.793L3 14.172V17h2.828l8.38-8.379-2.83-2.828z"/>
                                </svg>
                            </a>
                        </div>
                        
                        <div class="profile-preview">
                            <div class="profile-avatar-large" id="profileAvatar">U</div>
                            <div class="profile-info">
                                <h2 class="profile-name" id="profileName">Loading...</h2>
                                <p class="profile-desc" id="profileDesc">-</p>
                                <div class="profile-location" id="profileLocation">
                                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z"/>
                                    </svg>
                                    <span>-</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Contacts -->
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Recent Contacts</h3>
                        <a href="#" class="card-action" onclick="switchPage('contacts'); return false;">
                            View More →
                        </a>
                    </div>
                    
                    <div id="recentContactsContainer">
                        <div class="loading">
                            <div class="spinner"></div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Contacts Page -->
            <div id="contactsPage" style="display: none;">
                <div class="page-header">
                    <h1 class="page-title">Contacts 👥</h1>
                    <p class="page-subtitle">People who contacted you</p>
                </div>
                
                <div class="card">
                    <div id="allContactsContainer">
                        <div class="loading">
                            <div class="spinner"></div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Analytics Page -->
            <div id="analyticsPage" style="display: none;">
                <div class="page-header">
                    <h1 class="page-title">Analytics 📊</h1>
                    <p class="page-subtitle">Detailed statistics about your profile</p>
                </div>
                
                <div class="empty-state">
                    <div class="empty-state-icon">📊</div>
                    <p>Detailed analytics coming soon...</p>
                </div>
            </div>
            
            <!-- Settings Page -->
            <div id="settingsPage" style="display: none;">
                <div class="page-header">
                    <h1 class="page-title">Settings ⚙️</h1>
                    <p class="page-subtitle">Manage your account settings</p>
                </div>
                
                <div class="empty-state">
                    <div class="empty-state-icon">⚙️</div>
                    <p>Settings coming soon...</p>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        let userData = null;
        let currentPage = 'profile';
        
        // Check auth
        async function checkAuth() {
            const token = localStorage.getItem('user_token');
            if (!token) {
                window.location.href = 'user-login.php';
                return;
            }
            
            try {
                const response = await fetch('/api.php/user/me', {
                    headers: { 'Authorization': 'Bearer ' + token }
                });
                
                if (!response.ok) throw new Error('Unauthorized');
                
                userData = await response.json();
                updateUI();
                loadAnalytics();
                loadContacts();
            } catch (error) {
                localStorage.removeItem('user_token');
                window.location.href = 'user-login.php';
            }
        }
        
        function updateUI() {
            const initials = (userData.ime?.[0] || '') + (userData.prezime?.[0] || '');
            const fullName = `${userData.ime || ''} ${userData.prezime || ''}`.trim() || 'User';
            
            // Sidebar
            document.getElementById('sidebarAvatar').textContent = initials;
            document.getElementById('sidebarName').textContent = fullName;
            
            // Profile
            document.getElementById('profileAvatar').textContent = initials;
            document.getElementById('profileName').textContent = fullName;
            document.getElementById('profileDesc').textContent = userData.opis || 'No description';
            
            const location = [userData.grad, userData.drzava].filter(Boolean).join(', ') || 'No location';
            document.getElementById('profileLocation').querySelector('span').textContent = location;
        }
        
        async function loadAnalytics() {
            try {
                const token = localStorage.getItem('user_token');
                const response = await fetch('/api.php/user/analytics', {
                    headers: { 'Authorization': 'Bearer ' + token }
                });
                
                if (response.ok) {
                    const data = await response.json();
                    document.getElementById('statPageViews').textContent = data.total_views || 0;
                    document.getElementById('statClicks').textContent = data.total_clicks || 0;
                    document.getElementById('statCTR').textContent = (data.ctr || 0) + '%';
                }
            } catch (error) {
                console.error('Analytics error:', error);
            }
        }
        
        async function loadContacts() {
            try {
                const token = localStorage.getItem('user_token');
                const response = await fetch('/api.php/user/contacts?limit=5', {
                    headers: { 'Authorization': 'Bearer ' + token }
                });
                
                if (response.ok) {
                    const data = await response.json();
                    renderContacts(data.contacts || [], 'recentContactsContainer', true);
                } else {
                    document.getElementById('recentContactsContainer').innerHTML = 
                        '<div class="empty-state"><div class="empty-state-icon">📭</div><p>No contacts yet</p></div>';
                }
            } catch (error) {
                document.getElementById('recentContactsContainer').innerHTML = 
                    '<div class="empty-state"><div class="empty-state-icon">⚠️</div><p>Error loading contacts</p></div>';
            }
        }
        
        function renderContacts(contacts, containerId, isPreview = false) {
            const container = document.getElementById(containerId);
            
            if (!contacts || contacts.length === 0) {
                container.innerHTML = '<div class="empty-state"><div class="empty-state-icon">📭</div><p>No contacts yet</p></div>';
                return;
            }
            
            const html = `
                <table class="contacts-table">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Date</th>
                            <th>Tags</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${contacts.map(contact => `
                            <tr>
                                <td>
                                    <div style="display: flex; align-items: center; gap: 12px;">
                                        <div class="contact-avatar">${(contact.sender_name?.[0] || 'C').toUpperCase()}</div>
                                        <div>
                                            <div class="contact-name">${contact.sender_name || 'Unknown'}</div>
                                            <div class="contact-email">${contact.sender_email || '-'}</div>
                                        </div>
                                    </div>
                                </td>
                                <td>${new Date(contact.created_at).toLocaleDateString('hr-HR')}</td>
                                <td><span class="tag">New</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            
            container.innerHTML = html;
        }
        
        // Navigation
        function switchPage(page) {
            currentPage = page;
            
            // Update nav
            document.querySelectorAll('.nav-item').forEach(item => {
                item.classList.remove('active');
                if (item.dataset.page === page) {
                    item.classList.add('active');
                }
            });
            
            // Update pages
            document.getElementById('profilePage').style.display = page === 'profile' ? 'block' : 'none';
            document.getElementById('contactsPage').style.display = page === 'contacts' ? 'block' : 'none';
            document.getElementById('analyticsPage').style.display = page === 'analytics' ? 'block' : 'none';
            document.getElementById('settingsPage').style.display = page === 'settings' ? 'block' : 'none';
            
            // Load data if needed
            if (page === 'contacts') {
                loadAllContacts();
            }
        }
        
        async function loadAllContacts() {
            try {
                const token = localStorage.getItem('user_token');
                const response = await fetch('/api.php/user/contacts', {
                    headers: { 'Authorization': 'Bearer ' + token }
                });
                
                if (response.ok) {
                    const data = await response.json();
                    renderContacts(data.contacts || [], 'allContactsContainer', false);
                }
            } catch (error) {
                document.getElementById('allContactsContainer').innerHTML = 
                    '<div class="empty-state"><div class="empty-state-icon">⚠️</div><p>Error loading contacts</p></div>';
            }
        }
        
        // Nav click handlers
        document.querySelectorAll('.nav-item').forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                switchPage(item.dataset.page);
            });
        });
        
        function logout() {
            localStorage.removeItem('user_token');
            window.location.href = 'user-login.php';
        }
        
        // Init
        checkAuth();
    </script>
</body>
</html>
