<?php
// api.php - TapTap Card API

require_once 'config.php';

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// SECURITY: Rate limiting
$userIP = getUserIP();
checkRateLimit($userIP);

// SECURITY: Validate input size
if ($_SERVER['REQUEST_METHOD'] === 'POST' || $_SERVER['REQUEST_METHOD'] === 'PUT') {
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (strpos($contentType, 'application/json') !== false) {
        $rawInput = file_get_contents('php://input');
        if (strlen($rawInput) > 1048576) { // 1MB limit
            jsonResponse(['error' => 'Payload prevelik'], 413);
        }
    }
}

// Dohvati path iz URL-a
$request_uri = $_SERVER['REQUEST_URI'];
$uri_parts = parse_url($request_uri);
$path = isset($uri_parts['path']) ? trim($uri_parts['path'], '/') : '';

$path = str_replace('api.php/', '', $path);
$path = str_replace('api.php', '', $path);

$method = $_SERVER['REQUEST_METHOD'];

// ===== ROUTING =====

// Health check
if ($path === 'api/health' || $path === 'health') {
    jsonResponse(['status' => 'OK', 'message' => 'API je aktivan']);
}

// AUTH - Login
elseif (($path === 'api/auth/login' || $path === 'auth/login') && $method === 'POST') {
    $data = json_decode(file_get_contents('php://input'), true);
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';
    
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM admin_users WHERE username = ?');
    $stmt->execute([$username]);
    $admin = $stmt->fetch();
    
    if (!$admin || !password_verify($password, $admin['password'])) {
        jsonResponse(['error' => 'Pogrešni podaci'], 401);
    }
    
    $token = createToken($admin['id']);
    jsonResponse(['token' => $token, 'username' => $admin['username']]);
}

// USERS - Get all
elseif (($path === 'api/users' || $path === 'users') && $method === 'GET') {
    requireAuth();
    
    $db = getDB();
    $stmt = $db->query('SELECT * FROM users ORDER BY created_at DESC');
    $users = $stmt->fetchAll();
    
    jsonResponse($users);
}

// USERS - Get one by ID or slug
elseif (preg_match('#^(api/)?users/([a-zA-Z0-9_-]+)$#', $path, $matches) && $method === 'GET') {
    $identifier = $matches[2];
    
    $db = getDB();
    
    if (is_numeric($identifier)) {
        $stmt = $db->prepare('SELECT * FROM users WHERE id = ?');
        $stmt->execute([$identifier]);
    } else {
        $stmt = $db->prepare('SELECT * FROM users WHERE slug = ?');
        $stmt->execute([$identifier]);
    }
    
    $user = $stmt->fetch();
    
    if (!$user) {
        jsonResponse(['error' => 'Korisnik nije pronađen'], 404);
    }
    
    $stmt = $db->prepare('SELECT * FROM links WHERE user_id = ? ORDER BY sort_order');
    $stmt->execute([$user['id']]);
    $user['links'] = $stmt->fetchAll();
    
    $stmt = $db->prepare('SELECT * FROM modules WHERE user_id = ?');
    $stmt->execute([$user['id']]);
    $user['modules'] = $stmt->fetchAll();
    
    jsonResponse($user);
}

// USERS - Create or Update (POST sa _method=PUT)
elseif (preg_match('#^(api/)?users(/(\d+))?$#', $path, $matches) && $method === 'POST') {
    requireAuth();
    
    $isUpdate = isset($_POST['_method']) && $_POST['_method'] === 'PUT';
    $userId = $isUpdate ? ($_POST['user_id'] ?? ($matches[3] ?? null)) : null;
    
    if ($isUpdate && !$userId) {
        jsonResponse(['error' => 'Nedostaje user_id za update'], 400);
    }
    
    // Generiranje lozinke za analitiku (samo za nove korisnike)
    $analyticsPassword = null;
    if (!$isUpdate) {
        $analyticsPassword = $_POST['analytics_password'] ?? generatePassword(rand(6, 8));
    }
    
    $slug = $_POST['slug'] ?? '';
    
    // SECURITY: Validate slug
    if (!isValidSlug($slug)) {
        jsonResponse(['error' => 'Nevažeći slug format (3-50 znakova, samo slova, brojevi, - i _)'], 400);
    }
    
    $ime = $_POST['ime'] ?? '';
    $prezime = $_POST['prezime'] ?? '';
    $opis = $_POST['opis'] ?? '';
    $email = $_POST['email'] ?? '';
    $telefon = $_POST['telefon'] ?? '';
    $whatsapp = $_POST['whatsapp'] ?? '';
    $viber = $_POST['viber'] ?? '';
    $adresa = $_POST['adresa'] ?? '';
    $grad = $_POST['grad'] ?? '';
    $drzava = $_POST['drzava'] ?? '';
    $displayTemplate = $_POST['display_template'] ?? 'business';
    $accentColor = $_POST['accent_color'] ?? '#3b82f6';
    $backgroundType = $_POST['background_type'] ?? 'accent';
    $backgroundColor = $_POST['background_color'] ?? null;
    $analyticsKey = $_POST['analytics_key'] ?? null;
    $analyticsShortCode = $_POST['analytics_short_code'] ?? null;
    $active = isset($_POST['active']) ? (int)$_POST['active'] : 1;
    $activeFrom = !empty($_POST['active_from']) ? $_POST['active_from'] : null;
$activeUntil = !empty($_POST['active_until']) ? $_POST['active_until'] : null;
    
    // Upload profilne slike
    $slika = null;
    if (isset($_FILES['slika']) && $_FILES['slika']['error'] === UPLOAD_ERR_OK) {
        $ext = pathinfo($_FILES['slika']['name'], PATHINFO_EXTENSION);
        $filename = 'profile-' . time() . '-' . rand(1000, 9999) . '.' . $ext;
        
        if (!is_dir(UPLOAD_DIR)) mkdir(UPLOAD_DIR, 0755, true);
        
        if (move_uploaded_file($_FILES['slika']['tmp_name'], UPLOAD_DIR . $filename)) {
            $slika = '/uploads/' . $filename;
        }
    }
    
    // Upload background slike
    $bgImage = null;
    if (isset($_FILES['background_image']) && $_FILES['background_image']['error'] === UPLOAD_ERR_OK) {
        $ext = pathinfo($_FILES['background_image']['name'], PATHINFO_EXTENSION);
        $filename = 'bg-' . time() . '-' . rand(1000, 9999) . '.' . $ext;
        
        if (!is_dir(UPLOAD_DIR)) mkdir(UPLOAD_DIR, 0755, true);
        
        if (move_uploaded_file($_FILES['background_image']['tmp_name'], UPLOAD_DIR . $filename)) {
            $bgImage = '/uploads/' . $filename;
        }
    }
    
    $db = getDB();
    
    try {
        
        // Spremi redirect ako se slug mijenja
$oldSlugStmt = $db->prepare('SELECT slug FROM users WHERE id = ?');
$oldSlugStmt->execute([$userId]);
$oldSlug = $oldSlugStmt->fetchColumn();

if ($oldSlug && $oldSlug !== $slug) {
    $db->prepare('REPLACE INTO slug_redirects (old_slug, new_slug) VALUES (?, ?)')
       ->execute([$oldSlug, $slug]);
    
    // Regeneriraj .htaccess
    generateHtaccessRedirects($db);
}
        
        if ($isUpdate) {
            $sql = 'UPDATE users SET slug = ?, ime = ?, prezime = ?, opis = ?, email = ?, telefon = ?, whatsapp = ?, viber = ?, adresa = ?, grad = ?, drzava = ?, display_template = ?, accent_color = ?, background_type = ?, background_color = ?, analytics_key = ?, analytics_short_code = ?, active = ?, active_from = ?, active_until = ?';
$params = [$slug, $ime, $prezime, $opis, $email, $telefon, $whatsapp, $viber, $adresa, $grad, $drzava, $displayTemplate, $accentColor, $backgroundType, $backgroundColor, $analyticsKey, $analyticsShortCode, $active, $activeFrom, $activeUntil];

            if ($slika) {
                $sql .= ', slika = ?';
                $params[] = $slika;
            }
            
            if ($bgImage) {
                $sql .= ', background_image = ?';
                $params[] = $bgImage;
            }
            
            $sql .= ' WHERE id = ?';
            $params[] = $userId;
            
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
        } else {
            $stmt = $db->prepare('INSERT INTO users (slug, ime, prezime, opis, email, telefon, whatsapp, viber, adresa, grad, drzava, slika, display_template, accent_color, background_type, background_color, background_image, analytics_key, analytics_short_code, active, active_from, active_until) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([$slug, $ime, $prezime, $opis, $email, $telefon, $whatsapp, $viber, $adresa, $grad, $drzava, $slika, $displayTemplate, $accentColor, $backgroundType, $backgroundColor, $bgImage, $analyticsKey, $analyticsShortCode, $active, $activeFrom, $activeUntil]);
            
            $userId = $db->lastInsertId();
            
            // Spremi credentials za analytics
            if ($analyticsPassword) {
                $stmt = $db->prepare('INSERT INTO user_credentials (user_id, username, password) VALUES (?, ?, ?)');
                $stmt->execute([$userId, $slug, password_hash($analyticsPassword, PASSWORD_DEFAULT)]);
            }
        }
        
        // Update linkova
        if ($userId) {
            $stmt = $db->prepare('DELETE FROM links WHERE user_id = ?');
            $stmt->execute([$userId]);
            
            if (!empty($_POST['links'])) {
                $links = json_decode($_POST['links'], true);
                if ($links) {
                    foreach ($links as $index => $link) {
                        $stmt = $db->prepare('INSERT INTO links (user_id, tip, url, sort_order) VALUES (?, ?, ?, ?)');
                        $stmt->execute([$userId, $link['tip'], $link['url'], $index]);
                    }
                }
            }
            
            // Update modula
            if (!empty($_POST['modules'])) {
                $modules = json_decode($_POST['modules'], true);
                
                foreach (['mapa', 'instagram_feed', 'vcard', 'qr_code', 'text_block', 'google_reviews', 'working_hours'] as $moduleName) {
                    $aktivan = isset($modules[$moduleName]) && $modules[$moduleName] ? 1 : 0;
                    
                    $config = null;
                    if ($moduleName === 'mapa' && !empty($_POST['mapa_link'])) {
                        $config = json_encode(['link' => $_POST['mapa_link']]);
                    } elseif ($moduleName === 'instagram_feed' && !empty($_POST['instagram_username'])) {
                        $config = json_encode(['username' => $_POST['instagram_username']]);
                    } elseif ($moduleName === 'text_block' && !empty($_POST['text_block_content'])) {
                        $config = json_encode(['content' => $_POST['text_block_content']]);
                    } elseif ($moduleName === 'google_reviews' && !empty($_POST['google_reviews_link'])) {
                        $config = json_encode(['link' => $_POST['google_reviews_link']]);
                    } elseif ($moduleName === 'working_hours') {
                        $whType = $_POST['working_hours_type'] ?? 'all_days';
                        $whConfig = ['type' => $whType];
                        
                        if ($whType === 'all_days') {
                            $whConfig['all'] = $_POST['working_hours_all'] ?? '';
                        } else {
                            $whConfig['weekday'] = $_POST['working_hours_weekday'] ?? '';
                            $whConfig['saturday'] = $_POST['working_hours_saturday'] ?? '';
                            $whConfig['sunday'] = $_POST['working_hours_sunday'] ?? '';
                        }
                        
                        $config = json_encode($whConfig);
                    }
                    
                    $stmt = $db->prepare('INSERT INTO modules (user_id, modul_name, aktivan, config) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE aktivan = ?, config = ?');
                    $stmt->execute([$userId, $moduleName, $aktivan, $config, $aktivan, $config]);
                }
            }
        }
        
        if ($isUpdate) {
            jsonResponse(['message' => 'Korisnik ažuriran', 'userId' => $userId]);
        } else {
            jsonResponse(['id' => $userId, 'slug' => $slug], 201);
        }
        
    } catch (PDOException $e) {
        if ($e->getCode() == 23000) {
            jsonResponse(['error' => 'Slug već postoji'], 400);
        }
        jsonResponse(['error' => 'Greška: ' . $e->getMessage()], 500);
    }
}

// USERS - Delete
elseif (preg_match('#^(api/)?users/(\d+)$#', $path, $matches) && $method === 'DELETE') {
    requireAuth();
    
    $userId = $matches[2];
    $db = getDB();
    $stmt = $db->prepare('DELETE FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    
    jsonResponse(['message' => 'Korisnik obrisan']);
}

// LINKS - Create
elseif (preg_match('#^(api/)?users/(\d+)/links$#', $path, $matches) && $method === 'POST') {
    requireAuth();
    
    $userId = $matches[2];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = getDB();
    $stmt = $db->prepare('INSERT INTO links (user_id, tip, url, sort_order) VALUES (?, ?, ?, ?)');
    $stmt->execute([$userId, $data['tip'], $data['url'], $data['sort_order'] ?? 0]);
    
    jsonResponse(['id' => $db->lastInsertId()], 201);
}

// LINKS - Delete
elseif (preg_match('#^(api/)?links/(\d+)$#', $path, $matches) && $method === 'DELETE') {
    requireAuth();
    
    $linkId = $matches[2];
    $db = getDB();
    $stmt = $db->prepare('DELETE FROM links WHERE id = ?');
    $stmt->execute([$linkId]);
    
    jsonResponse(['message' => 'Link obrisan']);
}

// MODULES - Update
elseif (preg_match('#^(api/)?modules/(\d+)$#', $path, $matches) && $method === 'PUT') {
    requireAuth();
    
    $moduleId = $matches[2];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = getDB();
    $stmt = $db->prepare('UPDATE modules SET aktivan = ?, config = ? WHERE id = ?');
    $stmt->execute([
        $data['aktivan'] ? 1 : 0,
        $data['config'] ?? null,
        $moduleId
    ]);
    
    jsonResponse(['message' => 'Modul ažuriran']);
}

// QR CODE
elseif (preg_match('#^(api/)?qr/([a-zA-Z0-9_-]+)$#', $path, $matches) && $method === 'GET') {
    $slug = $matches[2];
    $url = BASE_URL . '/' . $slug . '?qr=1';
    
    $qrUrl = 'https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl=' . urlencode($url);
    
    jsonResponse([
        'qrCode' => $qrUrl,
        'url' => $url
    ]);
}

// VISITS - Log
elseif (preg_match('#^(api/)?visits/([a-zA-Z0-9_-]+)$#', $path, $matches) && $method === 'POST') {
    $slug = $matches[2];
    $data = json_decode(file_get_contents('php://input'), true);
    
    $db = getDB();
    $stmt = $db->prepare('SELECT id FROM users WHERE slug = ?');
    $stmt->execute([$slug]);
    $user = $stmt->fetch();
    
    if (!$user) {
        jsonResponse(['error' => 'Korisnik nije pronađen'], 404);
    }
    
    $ip = getUserIP();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $qrScan = $data['qr_scan'] ?? false;
    
    $stmt = $db->prepare('INSERT INTO visits (user_id, ip, user_agent, qr_scan) VALUES (?, ?, ?, ?)');
    $stmt->execute([$user['id'], $ip, $userAgent, $qrScan ? 1 : 0]);
    
    jsonResponse(['message' => 'Posjet zabilježen']);
}

// ANALYTICS - Admin (sa autentifikacijom)
elseif (preg_match('#^(api/)?analytics/(\d+)$#', $path, $matches) && $method === 'GET') {
    requireAuth();
    
    $userId = $matches[2];
    $db = getDB();
    
    $stmt = $db->prepare('SELECT COUNT(*) as total, SUM(qr_scan) as qr_scans FROM visits WHERE user_id = ?');
    $stmt->execute([$userId]);
    $stats = $stmt->fetch();
    
    $stmt = $db->prepare('SELECT ip, user_agent, timestamp, qr_scan FROM visits WHERE user_id = ? ORDER BY timestamp DESC LIMIT 50');
    $stmt->execute([$userId]);
    $recent = $stmt->fetchAll();
    
    jsonResponse([
        'total' => (int)$stats['total'],
        'qr_scans' => (int)($stats['qr_scans'] ?? 0),
        'recent' => $recent
    ]);
}

// ANALYTICS - Public (sa ključem)
elseif (preg_match('#^(api/)?analytics/([a-zA-Z0-9_-]+)/([a-zA-Z0-9]{32})$#', $path, $matches) && $method === 'GET') {
    $slug = $matches[2];
    $key = $matches[3];
    
    $db = getDB();
    $stmt = $db->prepare('SELECT id FROM users WHERE slug = ? AND analytics_key = ?');
    $stmt->execute([$slug, $key]);
    $user = $stmt->fetch();
    
    if (!$user) {
        jsonResponse(['error' => 'Neovlašteni pristup'], 403);
    }
    
    $userId = $user['id'];
    
    // Danas
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ? AND DATE(timestamp) = CURDATE()');
    $stmt->execute([$userId]);
    $today = $stmt->fetch();
    
    // Ovaj tjedan
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ? AND YEARWEEK(timestamp, 1) = YEARWEEK(CURDATE(), 1)');
    $stmt->execute([$userId]);
    $thisWeek = $stmt->fetch();
    
    // Ovaj mjesec
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ? AND YEAR(timestamp) = YEAR(CURDATE()) AND MONTH(timestamp) = MONTH(CURDATE())');
    $stmt->execute([$userId]);
    $thisMonth = $stmt->fetch();
    
    // Prošli mjesec
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ? AND YEAR(timestamp) = YEAR(CURDATE() - INTERVAL 1 MONTH) AND MONTH(timestamp) = MONTH(CURDATE() - INTERVAL 1 MONTH)');
    $stmt->execute([$userId]);
    $lastMonth = $stmt->fetch();
    
    // Ova godina
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ? AND YEAR(timestamp) = YEAR(CURDATE())');
    $stmt->execute([$userId]);
    $thisYear = $stmt->fetch();
    
    // Ukupno
    $stmt = $db->prepare('SELECT COUNT(*) as count, SUM(qr_scan) as qr FROM visits WHERE user_id = ?');
    $stmt->execute([$userId]);
    $total = $stmt->fetch();
    
    jsonResponse([
        'slug' => $slug,
        'today' => ['visits' => (int)$today['count'], 'qr_scans' => (int)($today['qr'] ?? 0)],
        'this_week' => ['visits' => (int)$thisWeek['count'], 'qr_scans' => (int)($thisWeek['qr'] ?? 0)],
        'this_month' => ['visits' => (int)$thisMonth['count'], 'qr_scans' => (int)($thisMonth['qr'] ?? 0)],
        'last_month' => ['visits' => (int)$lastMonth['count'], 'qr_scans' => (int)($lastMonth['qr'] ?? 0)],
        'this_year' => ['visits' => (int)$thisYear['count'], 'qr_scans' => (int)($thisYear['qr'] ?? 0)],
        'total' => ['visits' => (int)$total['count'], 'qr_scans' => (int)($total['qr'] ?? 0)]
    ]);
}

// ANALYTICS SHORT LINK - Redirect
elseif (preg_match('#^a/([a-zA-Z0-9_-]+)-([A-Z0-9]{7})$#', $path, $matches) && $method === 'GET') {
    $slug = $matches[1];
    $shortCode = $matches[2];
    
    $db = getDB();
    $stmt = $db->prepare('SELECT analytics_key FROM users WHERE slug = ? AND analytics_short_code = ?');
    $stmt->execute([$slug, $shortCode]);
    $user = $stmt->fetch();
    
    if (!$user || !$user['analytics_key']) {
        http_response_code(404);
        die('Link nije valjan');
    }
    
    header("Location: /{$slug}/analytics/{$user['analytics_key']}");
    exit;
}

// ANALYTICS - Universal login
elseif ($method === 'POST') {
    $analyticsSlug = getPlatformSetting('analytics_slug', 'analitika');
    
    // Provjeri je li path zapravo analytics endpoint
    $pathParts = explode('/', $path);
    $endpoint = end($pathParts);
    
    if ($endpoint === $analyticsSlug) {
        $data = json_decode(file_get_contents('php://input'), true);
        $username = $data['username'] ?? '';
        $password = $data['password'] ?? '';
        
        $db = getDB();
        $stmt = $db->prepare('SELECT uc.*, u.slug, u.analytics_key FROM user_credentials uc JOIN users u ON uc.user_id = u.id WHERE uc.username = ? AND uc.active = 1');
        $stmt->execute([$username]);
        $cred = $stmt->fetch();
        
        if (!$cred || !password_verify($password, $cred['password'])) {
            jsonResponse(['error' => 'Pogrešni podaci'], 401);
        }
        
        jsonResponse([
            'success' => true,
            'redirect' => "/{$cred['slug']}/analytics/{$cred['analytics_key']}"
        ]);
    }
}

// EXPORT - Single user
elseif (preg_match('#^(api/)?export/user/(\d+)$#', $path, $matches) && $method === 'GET') {
    requireAuth();
    
    $userId = $matches[2];
    $db = getDB();
    
    $stmt = $db->prepare('SELECT * FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        jsonResponse(['error' => 'Korisnik nije pronađen'], 404);
    }
    
    $stmt = $db->prepare('SELECT * FROM links WHERE user_id = ? ORDER BY sort_order');
    $stmt->execute([$userId]);
    $links = $stmt->fetchAll();
    
    $stmt = $db->prepare('SELECT * FROM modules WHERE user_id = ?');
    $stmt->execute([$userId]);
    $modules = $stmt->fetchAll();
    
    $export = [
        'version' => '1.0',
        'exported_at' => date('Y-m-d H:i:s'),
        'user' => $user,
        'links' => $links,
        'modules' => $modules
    ];
    
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="user-' . $user['slug'] . '-' . date('Y-m-d') . '.json"');
    echo json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// EXPORT - All users
elseif (($path === 'api/export/all' || $path === 'export/all') && $method === 'GET') {
    requireAuth();
    
    $db = getDB();
    
    $stmt = $db->query('SELECT * FROM users ORDER BY id');
    $users = $stmt->fetchAll();
    
    $exportData = [];
    
    foreach ($users as $user) {
        $stmt = $db->prepare('SELECT * FROM links WHERE user_id = ? ORDER BY sort_order');
        $stmt->execute([$user['id']]);
        $links = $stmt->fetchAll();
        
        $stmt = $db->prepare('SELECT * FROM modules WHERE user_id = ?');
        $stmt->execute([$user['id']]);
        $modules = $stmt->fetchAll();
        
        $exportData[] = [
            'user' => $user,
            'links' => $links,
            'modules' => $modules
        ];
    }
    
    $export = [
        'version' => '1.0',
        'exported_at' => date('Y-m-d H:i:s'),
        'users_count' => count($exportData),
        'users' => $exportData
    ];
    
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="all-users-' . date('Y-m-d') . '.json"');
    echo json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// IMPORT
elseif (($path === 'api/import' || $path === 'import') && $method === 'POST') {
    requireAuth();
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['version'])) {
        jsonResponse(['error' => 'Nevažeći format importa'], 400);
    }
    
    $db = getDB();
    $imported = 0;
    $errors = [];
    
    $usersToImport = isset($data['users']) ? $data['users'] : [$data];
    
    foreach ($usersToImport as $userData) {
        try {
            $user = $userData['user'];
            $links = $userData['links'] ?? [];
            $modules = $userData['modules'] ?? [];
            
            $stmt = $db->prepare('SELECT id FROM users WHERE slug = ?');
            $stmt->execute([$user['slug']]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                $errors[] = "Korisnik sa slugom '{$user['slug']}' već postoji - preskočen";
                continue;
            }
            
            $stmt = $db->prepare('INSERT INTO users (slug, ime, prezime, opis, email, telefon, whatsapp, viber, adresa, grad, drzava, slika, display_template, accent_color, background_type, background_color, background_image, analytics_key, analytics_short_code, active, active_from, active_until) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([
                $user['slug'],
                $user['ime'],
                $user['prezime'],
                $user['opis'] ?? null,
                $user['email'] ?? null,
                $user['telefon'] ?? null,
                $user['whatsapp'] ?? null,
                $user['viber'] ?? null,
                $user['adresa'] ?? null,
                $user['grad'] ?? null,
                $user['drzava'] ?? null,
                $user['slika'] ?? null,
                $user['display_template'] ?? 'business',
                $user['accent_color'] ?? '#3b82f6',
                $user['background_type'] ?? 'accent',
                $user['background_color'] ?? null,
                $user['background_image'] ?? null,
                $user['analytics_key'] ?? null,
                $user['analytics_short_code'] ?? null,
                $user['active'] ?? 1,
                $user['active_from'] ?? null,
                $user['active_until'] ?? null
            ]);
            
            $newUserId = $db->lastInsertId();
            
            foreach ($links as $link) {
                $stmt = $db->prepare('INSERT INTO links (user_id, tip, url, sort_order) VALUES (?, ?, ?, ?)');
                $stmt->execute([
                    $newUserId,
                    $link['tip'],
                    $link['url'],
                    $link['sort_order'] ?? 0
                ]);
            }
            
            foreach ($modules as $module) {
                $stmt = $db->prepare('INSERT INTO modules (user_id, modul_name, aktivan, config) VALUES (?, ?, ?, ?)');
                $stmt->execute([
                    $newUserId,
                    $module['modul_name'],
                    $module['aktivan'] ?? 0,
                    $module['config'] ?? null
                ]);
            }
            
            $imported++;
            
        } catch (Exception $e) {
            $errors[] = "Greška kod korisnika '{$user['slug']}': " . $e->getMessage();
        }
    }
    
    jsonResponse([
        'success' => true,
        'imported' => $imported,
        'errors' => $errors
    ]);
}

// SETTINGS - Get all
elseif (($path === 'api/settings' || $path === 'settings') && $method === 'GET') {
    requireAuth();
    
    $db = getDB();
    $stmt = $db->query('SELECT * FROM platform_settings');
    $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    jsonResponse($settings);
}

// SETTINGS - Update
elseif (($path === 'api/settings' || $path === 'settings') && $method === 'POST') {
    requireAuth();
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    foreach ($data as $key => $value) {
        savePlatformSetting($key, $value);
    }
    
    jsonResponse(['success' => true, 'message' => 'Postavke spremljene']);
}

// Not found
else {
    jsonResponse([
        'error' => 'Endpoint nije pronađen',
        'debug' => [
            'path' => $path,
            'method' => $method,
            'uri' => $request_uri
        ]
    ], 404);
}
?>